/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

		Add keydb objects to databases
		
        $Id: pgpKeyAdd.c,v 1.76.2.4 2001/05/30 18:33:12 hal Exp $
____________________________________________________________________________*/
#include "pgpConfig.h"

#include <errno.h>
#include <stdio.h>
#include <string.h>

#include "pgpDebug.h"
#include "pgpPktByte.h"
#include "pgpKeyPriv.h"
#include "pgpHash.h"
#include "pgpKeySpec.h"
#include "pgpMem.h"
#include "pgpEnv.h"
#include "pgpErrors.h"
#include "pgpPubKey.h"
#include "pgpFile.h"
#include "pgpMakeSig.h"
#include "pgpSigSpec.h"
#include "pgpContext.h"
#include "pgpTrustPriv.h"
#include "pgpX509Priv.h"

#ifndef NULL
#define NULL 0
#endif


/* Simple link manipulations */

/* Link key last in keydb */
	static void
pgpLinkKey( PGPKeyDBObj *key, PGPKeyDB *kdb )
{
	PGPKeyDBObj *lastkey;

	if( IsNull( kdb->firstKeyInDB ) )
	{
		kdb->firstKeyInDB = key;
	} else {
		lastkey = kdb->lastKeyInDB;;
		if( IsNull( lastkey ) )
			lastkey = kdb->firstKeyInDB;
		while( IsntNull( lastkey->next ) )
			lastkey = lastkey->next;
		lastkey->next = key;
	}
	key->next = NULL;
	key->up = (PGPKeyDBObj *)kdb;		/* Overload up pointer as db pointer */
	kdb->lastKeyInDB = key;
}

	static void
pgpUnlinkKey( PGPKeyDBObj *key, PGPKeyDB *kdb )
{
	PGPKeyDBObj *prev;

	if( kdb->firstKeyInDB == key )
	{
		kdb->firstKeyInDB = key->next;
		key->next = NULL;
		return;
	}
	for( prev = kdb->firstKeyInDB; IsntNull(prev->next); prev=prev->next )
	{
		if( prev->next == key )
		{
			prev->next = key->next;
			key->next = NULL;
			return;
		}
	}
	pgpAssert( 0 );
}

/* link at end; put userids before subkeys, sigs before them */
	static void
pgpLinkChild( PGPKeyDBObj *child, PGPKeyDBObj *parent )
{
	PGPKeyDBObj **pnext;
	PGPBoolean isUserID;
	PGPBoolean isSig;

	/* Put sigs first, then userids, then subkeys */
	isUserID = pgpObjectType(child) == RINGTYPE_USERID;
	isSig = pgpObjectType(child) == RINGTYPE_SIG;

	/* pnext points at pointer to next sibling */
	for( pnext = &parent->down; IsntNull( *pnext ); pnext = &((*pnext)->next) )
	{
		PGPInt32 nexttype = pgpObjectType(*pnext);
		if( isSig && nexttype!=RINGTYPE_SIG )
			break;
		if( isUserID && nexttype!=RINGTYPE_SIG && nexttype!=RINGTYPE_USERID)
			break;
	}
		
	child->next = *pnext;
	*pnext = child;
	child->up = parent;
}


	static void
pgpUnlinkChild( PGPKeyDBObj *child, PGPKeyDBObj *parent )
{
	PGPKeyDBObj *prev;

	if( parent->down == child )
	{
		parent->down = child->next;
		child->next = NULL;
		child->up = NULL;
		return;
	}
	for( prev = parent->down; IsntNull(prev->next); prev=prev->next )
	{
		if( prev->next == child )
		{
			prev->next = child->next;
			child->next = NULL;
			child->up = NULL;
			return;
		}
	}
	pgpAssert(0);
}


	static PGPKeyDBObj *
sNewObj( PGPKeyDB *kdb, PGPSize infosize )
{
	PGPKeyDBObjInfo *info;
	PGPKeyDBObj *obj;
	MemPool cutback;

	cutback = kdb->objPool;
	obj = memPoolAlloc( &kdb->objPool, sizeof(PGPKeyDBObj), sizeof (void *) );
	if (IsNull( obj ) )
		goto outofmem;
	info = memPoolAlloc( &kdb->objPool, infosize, sizeof (void *) );
	if (IsNull( info ) )
		goto outofmem;
	pgpClearMemory( obj, sizeof(*obj) );
	pgpClearMemory( info, infosize );

	obj->magic = kPGPKeyDBObjMagic;
	obj->idinfo.info = info;
	return obj;
 outofmem:
	memPoolCutBack( &kdb->objPool, &cutback );
	pgpKeyDBSetError( kdb, kPGPError_OutOfMemory );
	return NULL;
}


	static PGPKeyDBObj *
sNewKey( PGPKeyDB *kdb )
{
	PGPKeyDBObj *key;

	key = sNewObj( kdb, sizeof(PGPKeyInfo) );
	if( IsNull( key ) )
		return NULL;
	key->objflags = kPGPKeyDBObjType_Key;
	return key;
}

	static PGPKeyDBObj *
sNewUserID( PGPKeyDB *kdb )
{
	PGPKeyDBObj *userid;

	userid = sNewObj( kdb, sizeof(PGPUserIDInfo) );
	if( IsNull( userid ) )
		return NULL;
	userid->objflags = kPGPKeyDBObjType_UserID;
	return userid;
}

	static PGPKeyDBObj *
sNewSig( PGPKeyDB *kdb )
{
	PGPKeyDBObj *sig;

	sig = sNewObj( kdb, sizeof(PGPSigInfo) );
	if( IsNull( sig ) )
		return NULL;
	sig->objflags = kPGPKeyDBObjType_Signature;
	return sig;
}

	static PGPKeyDBObj *
sNewSubKey( PGPKeyDB *kdb )
{
	PGPKeyDBObj *subkey;

	subkey = sNewObj( kdb, sizeof(PGPKeyInfo) );
	if( IsNull( subkey ) )
		return NULL;
	subkey->objflags = kPGPKeyDBObjType_SubKey;
	return subkey;
}

	static PGPKeyDBObj *
sNewCRL( PGPKeyDB *kdb )
{
	PGPKeyDBObj *crl;

	crl = sNewObj( kdb, sizeof(PGPCRLInfo) );
	if( IsNull( crl ) )
		return NULL;
	crl->objflags = kPGPKeyDBObjType_CRL;
	return crl;
}

	static PGPKeyDBObj *
sNewUnknown( PGPKeyDB *kdb )
{
	PGPKeyDBObj *unk;

	unk = sNewObj( kdb, sizeof(PGPUnkInfo) );
	if( IsNull( unk ) )
		return NULL;
	return unk;
}

/*
 * X.509 certs require some special handling.
 *
 * Ordinary PGP signatures are linked to their issuing key by the
 * keyid field (and algorithm byte).  We maintain the sigsby/nextby
 * linked list which links keys to the signatures they have issued.
 * When we add a new signature, we look for a key with that keyid,
 * and create a Dummy one if not found.  We then add the sig to the
 * sigsby list for that key.  When we add a new key, if it already
 * exists nothing happens; if a Dummy version exists then its sigsby
 * list already points at the signatures which have been issued by this
 * key, so we clear the Dummy flag; and if there is no key with this
 * keyid, that means that there have been no signatures issued by this
 * key, so we just create a new key object and its sigsby list is
 * empty.
 *
 * X.509 certificates are linked to their issuing certificate (note,
 * certificate not key) by the Issuer Name field.  This is a
 * structured list of textual subfields like Common Name,
 * Organizational Unit, etc.  This introduces several problems for us.
 * First, we want to look for keys by keyid, not a set of text fields.
 * Second, when we first read an X.509 CA key, we may not know its
 * Issuer Name field, not until we read its self-signed CA
 * certificate.  Theoretically that could happen at some later time.
 * Third, if we create a dummy key to represent the signer for X.509
 * certificates, this would probably be a different dummy key than the
 * one created as a dummy for PGP keys.
 *
 * To handle this, we look up X.509 keys by creating a keyid-like
 * field which is the hash of the Issuer Name.  Every X.509 issuer has
 * two key object entries, the real one, and an X.509 Dummy whose
 * keyid is this hash.  The X.509 Dummy is tagged as such, and
 * overloads its data field to point at the X.509 cert on the real
 * key.  (Theoretically, an X.509 issuer could have more than one
 * X.509 Dummy associated with it, if it had more than one cert with
 * different Issuer Names.)
 *
 * The basic mechanism of the X.509 Dummy solves our problems, as
 * follows.  When we import an X.509 certificate, we hash the Issuer
 * Name to produce the X.509 Dummy keyid and look up a key with that
 * ID.  (We also use a fake algorithm byte of 0x80.)  If found, we
 * then look at its data field to see if there is a real key that
 * corresponds, and use that if so.  (This last step is done in
 * GetKeyByKeyID.)  Otherwise we use the X.509 Dummy for the by field
 * of the sig, creating one if necessary.  This ensures that all X.509
 * certs issued by the same key will point at the same X.509 Dummy if
 * we don't have the issuing cert.
 *
 * Furthermore, when we import an X.509 certificate, the Subject Name
 * of the cert would be the Issuer Name used if the parent key issues
 * any certs of its own.  (In the case of a self-signed CA cert,
 * the Subject Name and Issuer Name are the same.)  So in addition
 * to the processing above, we also use the Subject Name of the cert
 * to look for and/or create an X.509 Dummy corresponding to the parent
 * key.  Keep in mind, this is for use of the parent key as an X.509
 * cert issuer.  The previous paragraph was with respect to finding
 * the key which issued this X.509 cert, this describes mechanism to
 * find certs issued by this X.509 key.
 *
 * Using the Issuer Name we create an X.509 Dummy keyid and look up or
 * create an X.509 Dummy key.  If found, we set its data field to
 * point at the cert on the issuing key.  Further, we move any keys
 * from the X.509 Dummy's sigsby list to the sigsby list of the parent
 * key, and change those sigs' by pointers to point to the parent key.
 * This way, once we have read the X.509 issuing key's certificate,
 * the issued certificates point at the issuing key.
 *
 */


/* Find fake X.509 keyid as hash of issuer and subject names */
/* Also return flag saying whether cert has a CRL distribution point */
	static PGPError
sX509KeyIDBytes( PGPKeyDB *kdb, PGPByte const *buf, PGPSize len,
	PGPBoolean *hasdpoint, PGPByte issuerkeyidbytes[8],
	PGPByte subjectkeyidbytes[8])
{
	PGPContextRef		context;
	PGPASN_XTBSCertificate *xtbscert;
	PGPByte const *		certbuf;
	PGPSize				certlen;
	PGPByte const *		issuername;
	PGPSize				issuerlen;
	PGPByte const *		subjectname;
	PGPSize				subjectlen;
	PGPByte const *		dpoint;
	PGPByte				hash[20];
	PGPError			err;

	(void) len;
	if( IsntNull( hasdpoint ) )
		*hasdpoint = FALSE;
	context = PGPPeekKeyDBContext( kdb );

	certbuf = pgpSigFindNAISubSubpacket (buf,
					SIGSUBSUB_X509, 0, &certlen, NULL, NULL, NULL, NULL);
	
	pgpAssert (IsntNull( certbuf ) );
	
	/* Skip type, version bytes */
	pgpAssert (certbuf[0] == SIGSUBSUB_X509);
	certbuf += 3;
	certlen -= 3;

	/* Find issuer name in cert */
	err = pgpX509BufferToXTBSCert( context, certbuf, certlen, &xtbscert);
	if( IsPGPError( err ) )
		return err;
	issuername = xtbscert->issuer.val;
	issuerlen = xtbscert->issuer.len;
	subjectname = xtbscert->subject.val;
	subjectlen = xtbscert->subject.len;

	pgpFingerprint20HashBuf(context, issuername, issuerlen, hash);
	pgpCopyMemory( hash + sizeof(hash) - 8, issuerkeyidbytes, 8 );
	pgpFingerprint20HashBuf(context, subjectname, subjectlen, hash);
	pgpCopyMemory( hash + sizeof(hash) - 8, subjectkeyidbytes, 8 );

	if( IsntNull( hasdpoint ) )
	{
		dpoint =  pgpX509XTBSCertToDistPoint( xtbscert, NULL );
		if( dpoint != NULL ) {
			*hasdpoint = TRUE;
			PGPFreeData( (PGPByte *)dpoint );
		}
	}

	pgpX509FreeXTBSCert( context, xtbscert );
	return kPGPError_NoErr;
}


/* Having processed an X.509 sig, create or check for a dummy X509 key object
 * which will represent the "X.509 key id" of the key.  That is the
 * low 64 bits of the hash of the subject name.
 */
	static PGPError
sFixupX509ParentKey( PGPKeyDB *kdb, PGPKeyDBObj *sig,
	PGPByte x509subjectkeyidbytes[8] )
{
	PGPKeyDBObj *			key;
	PGPKeyDBObj *			x509key;
	PGPKeyInfo *			kinfo;
	PGPKeyInfo *			x509kinfo;
	PGPSigInfo *			sinfo;
	PGPSigInfo *			lastsiginfo = NULL;
	PGPKeyID				keyID;
	PGPByte					pkalg;

	key = sig;
	while( !OBJISTOPKEY( key ) )
		key = key->up;
	kinfo = pgpKeyToKeyInfo( key );

	/* Look up X.509 dummy key paired with this one */
	pkalg = 0x80;
	pgpNewKeyIDFromRawData( x509subjectkeyidbytes, pkalg, 8, &keyID );
	pgpGetKeyByKeyID( kdb, &keyID, TRUE, TRUE, &x509key );
	if( IsntNull( x509key ) )
	{
		/* If we have a real key this will already be pointing at it */
		if( !pgpKeyDBObjIsX509Dummy( x509key ) )
		{
			/*
			 * Ideally, at this point x509key is the same as key,
			 * and we are done.  It means that our key already has its
			 * paired dummy x509 key.  Conceivably it could be a
			 * different key, though.  That would mean that we have *
			 * two certs on different keys with the same subject name.
			 * This is unfortunate if one were to issue an X.509 certs
			 * itself, since there would be no way to know which one
			 * was the signer of such a cert.  We won't worry too much
			 * about that since it shouldn't happen in X.509 land and
			 * it won't cause any trouble if it doesn't happen.
			 */
/*			pgpAssert( x509key == key ); */
			return kPGPError_NoErr;
		}
		/*
		 * Here we have a dummy x509 entry.  This means that we have read
		 * one or more X.509 certs which used this key as the signer.
		 * They created this dummy entry and pointed their "by" pointers
		 * to it.  Now we must point this dummy to our real key, and
		 * move those sigs' "by" pointers to the real key too.
		 */
		/* Set paired data pointer */
		x509kinfo = pgpKeyToKeyInfo( x509key );
		pgpAssert( IsNull( x509kinfo->data ) );
		x509kinfo->data = (void *)sig;
		/* Fix "by" pointers in sigs */
		pgpAssert( IsntNull( x509kinfo->sigsby ) );
		for (sig = x509kinfo->sigsby; IsntNull(sig); sig = sinfo->nextby) {
			pgpAssert( OBJISSIG( sig ) );
			sinfo = pgpSigToSigInfo( sig );
			sinfo->by = key;
			lastsiginfo = sinfo;
		}
		/* Move sigs from x509kinfo->sigsby list to kinfo->sigsby list */
		lastsiginfo->nextby = kinfo->sigsby;
		kinfo->sigsby = x509kinfo->sigsby;
		x509kinfo->sigsby = NULL;
	} else {
		/*
		 * Here we need to create a dummy X.509 key entry which will
		 * be paired with our key and hold the alternate X509 keyid
		 * for our key.  */
		x509key = sNewKey( kdb );
		if( IsNull( key ) )
			return kPGPError_OutOfMemory;
		x509kinfo = pgpKeyToKeyInfo( x509key );
		pgpCopyMemory( x509subjectkeyidbytes, x509kinfo->keyID, 8 );
		x509kinfo->pkalg = pkalg;
		x509kinfo->data = (void *)sig;
		x509key->objflags |= kPGPKeyDBObjFlags_X509 | kPGPKeyDBObjFlags_Dummy;
		pgpLinkKey( x509key, kdb );
		pgpLinkKeyToTree( x509key, kdb );
	}
	return kPGPError_NoErr;
}

/*
 * Find paired CA key in another DB.  Will return X509 dummy unless a
 * "real" CA key exists in that DB.
 */
	PGPKeyDBObj *
pgpFindPairedX509Key( PGPKeyDBObj *cakey, PGPKeyDBRef kdb )
{
	PGPKeyDBObj *x509key;
	PGPKeyDBObj *casig;
	PGPKeyDBObj *child;
	PGPKeyDBObj *gchild;
	PGPByte const *sigbuf;
	PGPSize siglen;
	PGPByte issuerbytes[8];
	PGPByte subjectbytes[8];
	PGPKeyID keyID;
	PGPByte pkalg;

	pgpAssert( OBJISTOPKEY( cakey ) );

	/* Find CA signature so we can get subject name */
	casig = NULL;
	for( child = cakey->down; IsntNull(child) && IsNull(casig);
		 child = child->next )
	{
		if( !pgpKeyDBObjIsReal( child ) )
			continue;
		if( OBJISSIG( child ) && pgpSigIsX509( child ) )
		{
			casig = child;
			break;
		}
		for( gchild = child->down; IsntNull(gchild); gchild=gchild->next )
		{
			if( !pgpKeyDBObjIsReal( gchild ) )
				continue;
			if( OBJISSIG( gchild ) && pgpSigIsX509( gchild ) )
			{
				casig = gchild;
				break;
			}
		}
	}

	/* Probably casig shouldn't be able to be null here */
	if( casig == NULL )
		return NULL;
			
	sigbuf = pgpFetchObject( casig, &siglen );
	sX509KeyIDBytes( kdb, sigbuf, siglen, NULL, issuerbytes, subjectbytes );

	pkalg = 0x80;
	pgpNewKeyIDFromRawData( subjectbytes, pkalg, 8, &keyID );
	pgpGetKeyByKeyID( kdb, &keyID, TRUE, FALSE, &x509key );

	return x509key;
}


/*
 * See if object matches the given buffer.  Ignores differences in header
 * tag/length formatting.
 */
	static PGPBoolean
sObjectMatchesBuffer( PGPKeyDBObj const *obj, PGPByte const *buf, PGPSize len )
{
	PGPByte const *objbuf;
	PGPSize objlen;
	PGPSize objheaderlen, bufheaderlen;
	PGPByte buftype, objtype;

	/* Skip buffer past header */
	buftype = PKTBYTE_TYPE(*buf);
	bufheaderlen = pgpPktBufferHeaderLen( buf );
	buf += bufheaderlen;
	len -= bufheaderlen;

	objbuf = pgpFetchObject( obj, &objlen );
	objtype = PKTBYTE_TYPE(*objbuf);
	if( objtype != buftype )
		return FALSE;

	/* Skip object past header */
	objheaderlen = pgpPktBufferHeaderLen( objbuf );
	objbuf += objheaderlen;
	objlen -= objheaderlen;
	return objlen == len && pgpMemoryEqual( objbuf, buf, len );
}


/* Look for a matching object with specified parent.  Allow deleteds. */
	PGPKeyDBObj *
pgpFindMatchingChild( PGPKeyDBObj *parent, PGPByte const *buf, PGPSize len )
{
	PGPKeyDBObj *obj;

	for( obj=parent->down; IsntNull( obj ); obj = obj->next )
	{
		if( sObjectMatchesBuffer( obj, buf, len ) )
			break;
	}
	return obj;
}


/* True if data in a pubkey buffer is consistent with data in a seckey
 * buffer.  If evidence of version bug, return *vbug as true.  The version
 * bug happens if the sec buffer says V3 and the pub buffer says V2.
 */
	static PGPBoolean
sConsistentPubAndSec( PGPByte const *pdata, PGPSize plen, PGPByte const *sdata,
	PGPSize slen, PGPBoolean *vbug )
{
	PGPSize secsublen;
	PGPSize secheaderlen;
	PGPSize pubheaderlen;
	PGPByte secver;
	PGPByte pubver;

	if( IsntNull( vbug ) )
		*vbug = FALSE;

	pgpAssert( PKTBYTE_TYPE(pdata[0]) == PKTBYTE_PUBKEY ||
			   PKTBYTE_TYPE(pdata[0]) == PKTBYTE_PUBSUBKEY);
	pgpAssert( PKTBYTE_TYPE(sdata[0]) == PKTBYTE_SECKEY ||
			   PKTBYTE_TYPE(sdata[0]) == PKTBYTE_SECSUBKEY);

	secsublen = pgpKeyParsePublicPrefix( sdata, slen );
	secheaderlen = pgpPktBufferHeaderLen( sdata );
	secver = sdata[secheaderlen];
	pubheaderlen = pgpPktBufferHeaderLen( pdata );
	pubver = pdata[pubheaderlen];

	/* Compare all but version byte.  Return false if that doesn't match */
	if( secsublen-secheaderlen != plen-pubheaderlen ||
		!pgpMemoryEqual( sdata+secheaderlen+1, pdata+pubheaderlen+1,
						 secsublen-secheaderlen-1 ) )
		return FALSE;

	/* Okay, they matched.  Now compare versions. */
	if( secver == pubver )
		return TRUE;

	/* Versions disagreed, check for version bug, else return false */
	if( secver == PGPVERSION_3 && pubver == PGPVERSION_2 )
	{
		*vbug = TRUE;
		return TRUE;
	}
	return FALSE;
}


/*
 * Create a public key buffer from a given private key buffer.  Pubkey
 * buffer must be freed.
 */
	PGPByte *
pgpKeyDBObjToPubData( PGPKeyDBObj *key, PGPSize *retlen )
{
	PGPKeyDB *			keydb;
	PGPContextRef		context;
	PGPByte const *		sdata;
	PGPByte *			pdata;
	PGPSize				slen;
	PGPSize				plen;
	PGPSize				secsublen;
	PGPSize				secheaderlen;
	PGPSize				pubheaderlen;
	PGPKeyInfo *		kinfo;
	PGPByte				pubpkttype;

	pgpAssert( OBJISKEY( key ) );
	pgpAssert( pgpKeyIsSec( key ) );
	keydb = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( keydb );
	kinfo = pgpKeyToKeyInfo( key );

	sdata = pgpFetchObject( key, &slen );
	if( IsNull( sdata ) )
		return NULL;

	if( PKTBYTE_TYPE(sdata[0]) != PKTBYTE_SECKEY &&
		PKTBYTE_TYPE(sdata[0]) != PKTBYTE_SECSUBKEY )
	{
		/* This can happen if secret part is external, on a token */
		pgpAssert( PKTBYTE_TYPE(sdata[0]) == PKTBYTE_PUBKEY ||
				   PKTBYTE_TYPE(sdata[0]) == PKTBYTE_PUBSUBKEY );
		pdata = pgpContextMemAlloc( context, slen, 0 );
		if( IsNull( pdata ) )
			return NULL;
		pgpCopyMemory( sdata, pdata, slen );
		*retlen = slen;
		return pdata;
	}

	secsublen = pgpKeyParsePublicPrefix( sdata, slen );
	secheaderlen = pgpPktBufferHeaderLen( sdata );
	pubpkttype = (PKTBYTE_TYPE(sdata[0]) == PKTBYTE_SECKEY)? PKTBYTE_PUBKEY
														   : PKTBYTE_PUBSUBKEY;
	pubpkttype = PKTBYTE_BUILD( pubpkttype, 1 );
	pubheaderlen = pgpPktHeaderLen( pubpkttype, secsublen );
	plen = secsublen - secheaderlen + pubheaderlen;
	
	/* Allocate space for pub buffer */
	pdata = pgpContextMemAlloc( context, plen, 0 );
	if( IsNull( pdata ) )
		return NULL;
	
	/* Fill buffer */
	pgpPktHeaderPut( pdata, pubpkttype, secsublen-secheaderlen );
	pgpCopyMemory( sdata+secheaderlen, pdata+pubheaderlen,
				   secsublen-secheaderlen );
	
	/* Fix version bug */
	if( KEYHASVERSIONBUG(kinfo) )
		pdata[pubheaderlen] = PGPVERSION_2;

	*retlen = plen;
	return pdata;
}


PGPKeyDBObj *
pgpAddKey( PGPInt32 pkttype, PGPByte const *buf, PGPSize len,
	PGPKeyDB *kdb, PGPKeyDBObj *parent, PGPByte trusted, PGPError *outerr )
{
	PGPKeyDBObj *	key;
	PGPByte			pkalg;
	PGPByte			vpkalg;
	PGPUInt16		keybits;
	PGPByte			keyIDBytes[8];
	PGPByte			fp20n[20];
	PGPUInt32		tstamp;
	PGPUInt16		validity;
	PGPByte			v3;
	PGPBoolean		fSubkey;
	PGPBoolean		fSecret;
	PGPBoolean		fDummy;
	PGPByte const *	kdata;
	PGPSize			kdatalen;
	PGPKeyInfo *	kinfo;
	PGPBoolean		vbug;
	PGPKeyID		keyID;
	PGPError		err;

	fSubkey = (pkttype == PKTBYTE_PUBSUBKEY) || (pkttype == PKTBYTE_SECSUBKEY);
	fSecret = (pkttype == PKTBYTE_SECKEY) || (pkttype == PKTBYTE_SECSUBKEY);
	err = pgpKeyParse(kdb->context, buf, len, &pkalg, keyIDBytes,
					  fp20n, &keybits, &tstamp, &validity, &v3, fSecret );
	if( IsPGPError( err ) )
		return NULL;

	vpkalg = pkalg;			/* Viacrypt correction */
	if ((vpkalg | 1) == 3)
		vpkalg = 1;
	pgpNewKeyIDFromRawData( keyIDBytes, vpkalg, sizeof(keyIDBytes), &keyID );

	fSubkey = (pkttype == PKTBYTE_PUBSUBKEY) || (pkttype == PKTBYTE_SECSUBKEY);
	if( fSubkey ) {
		/* Subkey */
		pgpAssert( IsntNull( parent ) );
		pgpAssert( OBJISTOPKEY( parent ) );
	} else {
		pgpAssert( IsNull( parent ) );
	}

	/* Find the matching key structure */
	(void)pgpGetKeyByKeyID( kdb, &keyID, TRUE, TRUE, &key);

	fDummy = IsntNull( key ) && pgpKeyDBObjIsDummy( key );
	if( IsntNull( key ) && !fDummy ) {
		if( OBJISTOPKEY(key) == fSubkey )
		{
			/* Previous key and this key disagree on subkey-ness. Use prev. */
			/* Note that if prev is a subkey we will reject new userids
			 * as they can't go below subkeys.
			 */
			key->objflags &= ~kPGPKeyDBObjFlags_Deleted;
			return key;
		}
		/* Compare key with our data */
		kdata = pgpFetchObject( key, &kdatalen );
		if( !sObjectMatchesBuffer( key, buf, len ) )
		{
			/* Here what we are importing doesn't exactly match what we have */
			/* If previous was deleted, replace with new data */
			if( pgpKeyDBObjIsDeleted( key ) )
			{
				key->objflags &= ~kPGPKeyDBObjFlags_Deleted;
				goto fillin;
			}
			/* Else could just be a sec/pub disagreement */
			if( pkttype == PKTBYTE_TYPE(kdata[0]) )
			{
				/* No, that's not it, ignore new one */
				return key;
			}
			/* Here we disagree on sec/pub, see if they are otherwise same */
			if( fSecret )
			{
				pgpAssert( PKTBYTE_TYPE(kdata[0]) == PKTBYTE_PUBKEY ||
						   PKTBYTE_TYPE(kdata[0]) == PKTBYTE_PUBSUBKEY );
				if( !sConsistentPubAndSec( kdata, kdatalen, buf, len, &vbug ) )
				{
					/* Inconsistent, ignore new data */
					return key;
				}
				if( trusted && vbug )
				{
					kinfo = pgpKeyToKeyInfo( key );
					KEYSETVERSIONBUG( kinfo );
				}
				/* New sec data must replace pub */
				kinfo = pgpKeyToKeyInfo( key );
				kinfo->data = pgpNewObjData( kdb, buf, len );
				KEYSETSEC( kinfo );
			} else {
				pgpAssert( !fSecret );
				pgpAssert( PKTBYTE_TYPE(kdata[0]) == PKTBYTE_SECKEY  ||
						   PKTBYTE_TYPE(kdata[0]) == PKTBYTE_SECSUBKEY );
				if( !sConsistentPubAndSec( buf, len, kdata, kdatalen, &vbug ) )
				{
					/* Inconsistent, ignore new data */
					return key;
				}
				/* New pub data matches sec; record evidence of version bug */
				if( trusted && vbug )
				{
					kinfo = pgpKeyToKeyInfo( key );
					KEYSETVERSIONBUG( kinfo );
				}
			}
		} else {
			/* Match OK on key */
			key->objflags &= ~kPGPKeyDBObjFlags_Deleted;
		}
		return key;
	}
	if( fDummy ) {
		/* Key was a keyid only key */
		if( fSubkey && OBJISTOPKEY( key ) ) {
			/* Here we had a dummy top level key which turns out
			 * to be a subkey (can't happen unless subkeys sign) */
			/* Move it to proper position and mark as subkey. */
			pgpUnlinkKey( key, kdb );
			pgpLinkChild( key, parent );
			key->objflags &= ~kPGPKeyDBObjType_Key;
			key->objflags |= kPGPKeyDBObjType_SubKey;
		}
		key->objflags &= ~kPGPKeyDBObjFlags_Dummy;
		kinfo = pgpKeyToKeyInfo( key );
		/* Fix disagreement on pkalg for dummy key */
		if( kinfo->pkalg == kPGPPublicKeyAlgorithm_RSA &&
						pkalg == kPGPPublicKeyAlgorithm_RSASignOnly )
			kinfo->pkalg = kPGPPublicKeyAlgorithm_RSASignOnly;
		pgpAssert( kinfo->pkalg == pkalg );
		pgpAssert( pgpMemoryEqual( kinfo->keyID, keyIDBytes,
								   sizeof(keyIDBytes) ) );
	}
	if( IsNull( key ) ) {
		key = fSubkey ? sNewSubKey( kdb ) : sNewKey( kdb );
		if ( key == NULL )
		{
			err = kPGPError_OutOfMemory;
			goto error;
		}
		if( fSubkey ) {
			pgpLinkChild( key, parent );
		} else {
			pgpLinkKey( key, kdb );
		}
		kinfo = pgpKeyToKeyInfo( key );
		kinfo->pkalg = pkalg;
		pgpCopyMemory( keyIDBytes, kinfo->keyID, sizeof(keyIDBytes) );
		pgpLinkKeyToTree( key, kdb );
	}
 fillin:
	/* Fill in key data */
	kinfo = pgpKeyToKeyInfo( key );
	kinfo->fp20n = fp20n[0];
	kinfo->creationtime = tstamp;
	kinfo->validityperiod = validity;
	kinfo->keybits = keybits;
	kinfo->trust = 0;
	if (v3)
		KEYSETV3(kinfo);
	else
		KEYCLEARV3(kinfo);
	if( fSecret )
		KEYSETSEC(kinfo);
	else
		KEYCLEARSEC(kinfo);
	kinfo->data = pgpNewObjData( kdb, buf, len );
	return key;
 error:
	*outerr = err;
	return NULL;
}


PGPKeyDBObj *
pgpAddUserID( PGPInt32 pkttype, PGPByte const *buf, PGPSize len,
	PGPKeyDB *kdb, PGPKeyDBObj *parent, PGPError *outerr )
{
	PGPKeyDBObj *		name;
	PGPUserIDInfo *		uinfo;
	PGPError			err;

	(void)pkttype;
	pgpAssert( IsntNull( parent ) );
	if( !OBJISTOPKEY( parent ) )
		/* Can't add it */
		return NULL;

	/* Find the matching name structure */
	name = pgpFindMatchingChild( parent, buf, len );

	if( IsntNull( name ) ) {
		/* Got a match */
		if( name->objflags & kPGPKeyDBObjFlags_Deleted )
		{
			/* Bring back to life at end of list */
			name->objflags &= ~kPGPKeyDBObjFlags_Deleted;
			pgpUnlinkChild( name, parent );
			pgpLinkChild( name, parent );
		}
		return name;
	} else {
		name = sNewUserID( kdb );
		if ( name == NULL )
		{
			err = kPGPError_OutOfMemory;
			goto error;
		}
		uinfo = pgpUserIDToUserIDInfo( name );
		uinfo->validity = PGP_NEWTRUST_UNDEFINED;
		uinfo->confidence = PGP_NEWTRUST_UNDEFINED;
		pgpLinkChild( name, parent );
	}
	uinfo = pgpUserIDToUserIDInfo( name );
	if( pkttype == PKTBYTE_ATTRIBUTE )
		NAMESETATTR(uinfo);
	uinfo->data = pgpNewObjData( kdb, buf, len );
	return name;
 error:
	*outerr = err;
	return NULL;
}

PGPKeyDBObj *
pgpAddSig( PGPInt32 pkttype, PGPByte const *buf, PGPSize len,
	PGPKeyDB *kdb, PGPKeyDBObj *parent, PGPError *error )
{
	PGPKeyDBObj *		sig;
	PGPKeyDBObj *		key;
	PGPKeyDBObj *		tsig;
	PGPKeyInfo *		kinfo;
	PGPSigInfo *		sinfo;
	PGPByte				pkalg;
	PGPByte				vpkalg;
	PGPByte				keyidbytes[8];
	PGPByte				x509subjectkeyidbytes[8];
	PGPUInt32			tstamp;
	PGPUInt32			sigvalidity;
	PGPByte				type;
	PGPByte				hashalg;
	PGPSize				extralen;
	PGPByte				version;
	PGPBoolean			exportable;
	PGPBoolean			revocable;
	PGPByte				trustLevel;
	PGPByte				trustValue;
	PGPBoolean			hasRegExp;
	PGPBoolean			hasDPoint = FALSE;
	PGPBoolean			isX509;
	PGPBoolean			primaryUID;
	PGPKeyID			keyID;
	PGPError			err;

	(void)pkttype;
	pgpAssert( IsntNull( parent ) );

	/* Look for matching sig structure */
	sig = pgpFindMatchingChild( parent, buf, len );

	if( IsntNull( sig ) ) {
		/* Got a match */
		if( sig->objflags & kPGPKeyDBObjFlags_Deleted )
		{
			/* Bring back to life at end of list */
			sig->objflags &= ~kPGPKeyDBObjFlags_Deleted;
			pgpUnlinkChild( sig, parent );
			pgpLinkChild( sig, parent );
			/* See if sig is on sigsby list, put it there if not */
			sinfo = pgpSigToSigInfo( sig );
			key = sinfo->by;
			kinfo = pgpKeyToKeyInfo( key );
			for( tsig = kinfo->sigsby; IsntNull(tsig); tsig=sinfo->nextby )
			{
				if ( tsig == sig )
					return sig;		/* Done */
				sinfo = pgpSigToSigInfo( tsig );
			}
			/* Not found, put it on list */
			sinfo = pgpSigToSigInfo( sig );
			sinfo->nextby = kinfo->sigsby;
			kinfo->sigsby = sig;
		}
		return sig;
	} else {
		err = pgpSigParse(buf, len, FALSE, &pkalg, keyidbytes, &tstamp,
						  &sigvalidity, &type, &hashalg, &extralen, &version,
						  &exportable, &revocable, &trustLevel, &trustValue,
						  &hasRegExp, &isX509, &primaryUID);
		if( IsPGPError( err ) )
			return NULL;

		/* X.509 sigs use hash of issuer name as keyid; also get hash of
		 * subject name for later. */
		if( isX509 )
		{
			sX509KeyIDBytes( kdb, buf, len, &hasDPoint,
							 keyidbytes, x509subjectkeyidbytes);
		}
		if( isX509 )			/* Use high values for pkalg on X509's */
			pkalg = 0x80;
		vpkalg = pkalg;			/* Viacrypt correction */
		if ((vpkalg | 1) == 3)
			vpkalg = 1;
		pgpNewKeyIDFromRawData(keyidbytes, vpkalg, sizeof(keyidbytes), &keyID);
		(void)pgpGetKeyByKeyID( kdb, &keyID, TRUE, TRUE, &key);
		if( IsNull( key ) ) {
			/* Must create dummy key */
			key = sNewKey( kdb );
			if( IsNull( key ) )
			{
				err = kPGPError_OutOfMemory;
				goto error;
			}
			kinfo = pgpKeyToKeyInfo( key );
			pgpCopyMemory( keyidbytes, kinfo->keyID, sizeof(keyidbytes));
			kinfo->pkalg = pkalg;
			key->objflags |= kPGPKeyDBObjFlags_Dummy;
			if( isX509 )
				key->objflags |= kPGPKeyDBObjFlags_X509;
			pgpLinkKey( key, kdb );
			pgpLinkKeyToTree( key, kdb );
		} else if( pgpKeyDBObjIsDeleted( key ) ) {
			key->objflags &= ~kPGPKeyDBObjFlags_Deleted;
			key->objflags |= kPGPKeyDBObjFlags_Dummy;
		}

		sig = sNewSig( kdb );
		if ( sig == NULL ) {
			err = kPGPError_OutOfMemory;
			goto error;
		}
		pgpLinkChild( sig, parent );

		sinfo = pgpSigToSigInfo( sig );
		sinfo->by = key;
		kinfo = pgpKeyToKeyInfo( key );
		sinfo->nextby = kinfo->sigsby;
		kinfo->sigsby = sig;
		if (extralen != 5)
			sinfo->flags |= SIGF_NONFIVE;
		sinfo->tstamp = tstamp;
		sinfo->sigvalidity = sigvalidity;
		sinfo->type = type;
		sinfo->version = version;
		sinfo->hashalg = hashalg;
		sinfo->trust = 0;
		sinfo->trustLevel = trustLevel;
		sinfo->trustValue = trustValue;
		if (exportable)
			SIGSETEXPORTABLE(sinfo);
		else
			SIGSETNONEXPORTABLE(sinfo);
		if (revocable)
			SIGSETREVOCABLE(sinfo);
		else
			SIGSETNONREVOCABLE(sinfo);
		if (hasRegExp)
			SIGSETUSESREGEXP(sinfo);
		else
			SIGCLEARUSESREGEXP(sinfo);
		if (isX509)
			SIGSETX509(sinfo);
		else
			SIGCLEARX509(sinfo);
		if (hasDPoint)
			SIGSETDISTPOINT(sinfo);
		else
			SIGCLEARDISTPOINT(sinfo);
		if (primaryUID)
			SIGSETPRIMARYUID(sinfo);
		else
			SIGCLEARPRIMARYUID(sinfo);
	}
	if( isX509 )
		sFixupX509ParentKey( kdb, sig, x509subjectkeyidbytes );
	sinfo = pgpSigToSigInfo( sig );

	if( IS_OLD_PKTBYTE(buf[0])
		&& PKTBYTE_LLEN(buf[0]) == 0
		&& pkalg == kPGPPublicKeyAlgorithm_RSA )
	{
		/* PGP 2.6.2 requires RSA sigs to have two byte lengths */
		PGPByte pktbyte;
		PGPByte *pktbuf;
		PGPContextRef context;

		pktbyte = PKTBYTE_BUILD( PKTBYTE_SIG, 1 );
		/* Len will be 1 bigger */
		len += 1;

		/* Allocate space */
		context = PGPPeekKeyDBContext( kdb );
		pktbuf = pgpContextMemAlloc( context, len, 0 );
		if( IsNull(pktbuf) )
			return NULL;

		/* Fill buffer.  len-3 is length of payload. */
		pgpPktHeaderPut( pktbuf, pktbyte, len-3 );
		pgpCopyMemory( buf+2, pktbuf+3, len-3 );

		/* Store data buffer */
		sinfo->data = pgpNewObjData( kdb, pktbuf, len );
		PGPFreeData( pktbuf );
		
	} else
		sinfo->data = pgpNewObjData( kdb, buf, len );

	return sig;
 error:
	*error = err;
	return NULL;
}


PGPKeyDBObj *
pgpAddCRL( PGPInt32 pkttype, PGPByte const *buf, PGPSize len,
	PGPKeyDB *kdb, PGPKeyDBObj *parent, PGPError *outerr )
{
	PGPKeyDBObj *		crl;
	PGPCRLInfo *		cinfo;
	PGPContextRef		context;
	PGPUInt32			tstamp, tstampnext;
	PGPByte				version;
	PGPByte				type;
	PGPByte const *		dpoint;
	PGPSize				dpointlen;
	PGPError			err;

	(void)pkttype;
	pgpAssert( IsntNull( parent ) );
	if( !OBJISTOPKEY( parent ) )
		return NULL;

	/* Find any matching CRL structure */
	crl = pgpFindMatchingChild( parent, buf, len );

	if( IsntNull( crl ) ) {
		/* Got a match */
		crl->objflags &= ~kPGPKeyDBObjFlags_Deleted;
		return crl;
	} else {
		crl = sNewCRL( kdb );
		if ( crl == NULL ) {
			err = kPGPError_OutOfMemory;
			goto error;
		}
		pgpLinkChild( crl, parent );
	}
	cinfo = pgpCRLToCRLInfo( crl );
	context = PGPPeekKeyDBContext( kdb );
	(void)pgpCRLParse( context, buf, len, &version, &type, &tstamp,
					   &tstampnext, &dpoint, &dpointlen );
	cinfo->version = version;
	cinfo->tstamp = tstamp;
	cinfo->tstampnext = tstampnext;
	if (type == PGPCRLTYPE_X509 || type == PGPCRLTYPE_X509DPOINT)
		CRLSETX509(cinfo);
	else
		CRLCLEARX509(cinfo);
	if (type == PGPCRLTYPE_X509DPOINT) {
		CRLSETDPOINT(cinfo);
	} else {
		CRLCLEARDPOINT(cinfo);
	}
	cinfo->data = pgpNewObjData( kdb, buf, len );
	return crl;
 error:
	*outerr = err;
	return NULL;
}

PGPKeyDBObj *
pgpAddUnknown( PGPInt32 pkttype, PGPByte const *buf, PGPSize len,
	PGPKeyDB *kdb, PGPKeyDBObj *parent, PGPError *outerr )
{
	PGPKeyDBObj *		unk;
	PGPUnkInfo *		ninfo;
	PGPError			err;

	(void)pkttype;
	pgpAssert( IsntNull( parent ) );
	if( !OBJISTOPKEY( parent ) )
		return NULL;

	/* Find any matching Unknown structure */
	unk = pgpFindMatchingChild( parent, buf, len );

	if( IsntNull( unk ) ) {
		/* Got a match */
		unk->objflags &= ~kPGPKeyDBObjFlags_Deleted;
		return unk;
	} else {
		unk = sNewUnknown( kdb );
		if ( unk == NULL ) {
			err = kPGPError_OutOfMemory;
			goto error;
		}
		pgpLinkChild( unk, parent );
	}
	ninfo = pgpUnkToUnkInfo( unk );
	ninfo->data = pgpNewObjData( kdb, buf, len );
	return unk;
 error:
	*outerr = err;
	return NULL;
}




PGPKeyDBObj *
pgpAddObject( PGPByte const *buf, PGPSize len,
	PGPKeyDB *kdb, PGPKeyDBObj *parent, PGPBoolean trusted, PGPError *err )
{
	PGPByte pkttype;

	if( len == 0 )
		return NULL;

	pkttype = PKTBYTE_TYPE(*buf);
	switch (pkttype) {
	case PKTBYTE_PUBKEY:
	case PKTBYTE_PUBSUBKEY:
	case PKTBYTE_SECKEY:
	case PKTBYTE_SECSUBKEY:
		return pgpAddKey( pkttype, buf, len, kdb, parent, trusted, err );
	case PKTBYTE_ATTRIBUTE:
	case PKTBYTE_USERID:
		return pgpAddUserID( pkttype, buf, len, kdb, parent, err );
	case PKTBYTE_SIG:
		return pgpAddSig( pkttype, buf, len, kdb, parent, err );
	case PKTBYTE_CRL:
		return pgpAddCRL( pkttype, buf, len, kdb, parent, err );
	default:
		return pgpAddUnknown( pkttype, buf, len, kdb, parent, err );
	}
}


static PGPError
pgpTrustPacket( PGPByte const *buf, PGPSize len, PGPKeyDB *kdb,
				PGPKeyDBObj *lastobj, PGPEnv *env )
{
	PGPKeyInfo *kinfo;
	PGPUserIDInfo *uinfo;
	PGPSigInfo *sinfo;
	PGPCRLInfo *cinfo;
	PGPUnkInfo *ninfo;
	PGPByte trust;
	PGPByte validity;
	PGPByte confidence;
	PGPContextRef context;
	PGPSize hlen;
	PGPEnv *pgpEnv;

	if( len == 0 )
		return kPGPError_NoErr;

	context = PGPPeekKeyDBContext( kdb );
	pgpEnv = pgpContextGetEnvironment( context );

	hlen = pgpPktBufferHeaderLen( buf );
	buf += hlen;
	len -= hlen;

	if( len == 0 )
		return kPGPError_NoErr;

	trust = buf[0];

	if( OBJISUSERID( lastobj ) ) {
		/*
		 * Name trust packets can be up to 3 bytes
		 * long:
		 * Byte 1: old-style trust packet
		 * Byte 2: validity of name
		 * Byte 3: confidence in name as introducer
		 */
		/* Default trust and validity bytes */
		switch (trust & kPGPUserIDTrust_Mask) {
		  case kPGPUserIDTrust_Complete:
			validity = pgpenvCompleteTrust( env );
			break;
		  case kPGPUserIDTrust_Marginal:
			validity = pgpenvMarginalTrust( env );
			break;
		  default:
			validity = 0;
			break;
		}
		pgpAssert(OBJISKEY(lastobj->up));

		kinfo = pgpKeyToKeyInfo( lastobj->up );
		confidence = pgpTrustOldToExtern(pgpEnv, kinfo->trust);

		if (len > 1) {
			/* Fetch validity */
			validity = buf[1];
			if (len > 2) {
				/* Fetch confidence */
				confidence = buf[2];
			}
		}
		uinfo = pgpUserIDToUserIDInfo( lastobj );
		uinfo->oldvalidity = trust;
		uinfo->validity = validity;
		uinfo->confidence = confidence;
		uinfo->valid = pgpTrustToIntern (validity);
	} else {	/* Not a name */
		/* Set the appropriate flags */
		switch (pgpObjectType(lastobj)) {
		default:
			pgpAssert(0);
		case RINGTYPE_KEY:
		case RINGTYPE_SUBKEY:
			kinfo = pgpKeyToKeyInfo( lastobj );
			kinfo->trust = trust;
			break;
		case RINGTYPE_SIG:
			sinfo = pgpSigToSigInfo( lastobj );
			sinfo->trust = trust;
			break;
		case RINGTYPE_CRL:
			cinfo = pgpCRLToCRLInfo( lastobj );
			cinfo->trust = trust;
			break;
		case RINGTYPE_UNK:
			ninfo = pgpUnkToUnkInfo( lastobj );
			ninfo->trust = trust;
			break;
		}
	}


	/* Handle key validity info from trusted self-signatures */
	if( OBJISSIG(lastobj) ) {
		PGPKeyDBObj *keyobj;
		PGPKeyDBObj *topkeyobj;
		PGPByte const *sdata;
		PGPSize sdatalen;
		PGPByte const *pk;

		sinfo = pgpSigToSigInfo( lastobj );
		if( (sinfo->trust & PGP_SIGTRUSTF_CHECKED) &&
			!(sinfo->trust & PGP_SIGTRUSTF_REVOKEDBYCRL) ) {
			keyobj = lastobj->up;
			while( !OBJISKEY( keyobj ) )
				keyobj = keyobj->up;
			topkeyobj = keyobj;
			while( !OBJISTOPKEY( topkeyobj ) )
				topkeyobj = topkeyobj->up;
			if( sinfo->by == topkeyobj ) {
				sdata = pgpFetchObject( lastobj, &sdatalen );
				pk = pgpSigFindSubpacket ( sdata,
					SIGSUB_KEY_EXPIRATION,
					0, NULL, NULL, NULL, NULL, NULL);
				if( IsntNull(pk) ) {
					PGPUInt32 keyexp;
					keyexp = (PGPUInt32)((unsigned)pk[0]<<8|pk[1]) << 16 |
									((unsigned)pk[2]<<8|pk[3]);
					kinfo = pgpKeyToKeyInfo( keyobj );
					kinfo->validityperiod = keyexp / (24 * 3600);
				}
			}
		}
	}
	return kPGPError_NoErr;
}


/*
 * When copying an obj from one db to another, copy its trust info.  We will
 * keep the max of the old and new trust.
 */
	void
pgpCopyObjectTrust( PGPKeyDBObj *oldobj, PGPKeyDBObj *newobj )
{
	PGPKeyInfo *okinfo, *nkinfo;
	PGPUserIDInfo *ouinfo, *nuinfo;
	PGPSigInfo *osinfo, *nsinfo;
	PGPCRLInfo *ocinfo, *ncinfo;
	PGPUnkInfo *oninfo, *nninfo;

	if( OBJISUSERID( oldobj ) ) {
		pgpAssert( OBJISUSERID( newobj ) );
		ouinfo = pgpUserIDToUserIDInfo( oldobj );
		nuinfo = pgpUserIDToUserIDInfo( newobj );
		nuinfo->oldvalidity =
			((ouinfo->oldvalidity|nuinfo->oldvalidity) &
			 (PGP_USERIDTRUSTF_WARNONLY|PGP_USERIDTRUSTF_PRIMARY)) |
			pgpMax(ouinfo->oldvalidity&3, nuinfo->oldvalidity&3);
		nuinfo->validity = (nuinfo->validity == PGP_NEWTRUST_UNDEFINED)
			? ouinfo->validity
			: ( (ouinfo->validity == PGP_NEWTRUST_UNDEFINED)
				? nuinfo->validity
				: pgpMax(ouinfo->validity, nuinfo->validity) );
		nuinfo->confidence = (nuinfo->confidence == PGP_NEWTRUST_UNDEFINED)
			? ouinfo->confidence
			: ( (ouinfo->confidence == PGP_NEWTRUST_UNDEFINED)
				? nuinfo->confidence
				: pgpMax(ouinfo->confidence, nuinfo->confidence) );
		nuinfo->valid = pgpMax(ouinfo->valid, nuinfo->valid);
	} else {	/* Not a name */
		/* Set the appropriate flags */
		switch (pgpObjectType(oldobj)) {
		default:
			pgpAssert(0);
		case RINGTYPE_KEY:
		case RINGTYPE_SUBKEY:
			pgpAssert(OBJISKEY(newobj));
			okinfo = pgpKeyToKeyInfo( oldobj );
			nkinfo = pgpKeyToKeyInfo( newobj );
			nkinfo->trust = pgpMax(okinfo->trust, nkinfo->trust);
			nkinfo->validityperiod = pgpMax(okinfo->validityperiod,
											nkinfo->validityperiod);
			/* Copy residence on crypto token */
			if( okinfo->flags & KEYF_TOKEN )
				nkinfo->flags |= KEYF_TOKEN;
			break;
		case RINGTYPE_SIG:
			pgpAssert(OBJISSIG(newobj));
			osinfo = pgpSigToSigInfo( oldobj );
			nsinfo = pgpSigToSigInfo( newobj );
			if( (osinfo->trust & PGP_SIGTRUSTF_CHECKED) &&
				!(nsinfo->trust & PGP_SIGTRUSTF_CHECKED) )
			{
				/* Newly trusted signature */
				if( SIGISPRIMARYUID (nsinfo) )
				{
					/* Clear any manual primary uid settings we had */
					PGPKeyDBObjRef key = PGPPeekKeyDBObjKey( newobj );
					if( pgpSigMaker( newobj ) == key )
						pgpKeyClearPrimaryUserIDs( key, (PGPUInt32)-1UL );
				}
			}
			nsinfo->trust = pgpMax(osinfo->trust, nsinfo->trust);
			break;
		case RINGTYPE_CRL:
			pgpAssert(OBJISCRL(newobj));
			ocinfo = pgpCRLToCRLInfo( oldobj );
			ncinfo = pgpCRLToCRLInfo( newobj );
			ncinfo->trust = pgpMax(ocinfo->trust, ncinfo->trust);
			break;
		case RINGTYPE_UNK:
			pgpAssert(pgpObjectType(newobj) == RINGTYPE_UNK);
			oninfo = pgpUnkToUnkInfo( oldobj );
			nninfo = pgpUnkToUnkInfo( newobj );
			nninfo->trust = pgpMax(oninfo->trust, nninfo->trust);
			break;
		}
	}
}


static PGPError
sCleanupThirdPartyRevocations(PGPKeyDBObj *key, PGPBoolean trusted)
{
	PGPKeyDBObj *rsig, *nextrsig;
	PGPKeyDBObj *ssig;
	PGPSigInfo *rsinfo, *ssinfo;
	PGPKeyInfo *rskinfo;
	PGPBoolean success;
	PGPUInt32 i;
	PGPUInt32 matches;
	PGPByte const *p;
	PGPSize len;

	(void) trusted;
	for( rsig=key->down; IsntNull(rsig); rsig=nextrsig )
	{
		nextrsig = rsig->next;
		if( !OBJISSIG(rsig) )
			continue;
		rsinfo = pgpSigToSigInfo( rsig );
		if( rsinfo->type != PGP_SIGTYPE_KEY_REVOKE )
			continue;
		if( rsinfo->by == key )
			continue;
		rskinfo = pgpKeyToKeyInfo( rsinfo->by );
		/* Here we have a third party revocation signature */
		/* See if there is a self-sig on the key that authorizes it */
		success = FALSE;
		for( ssig=key->down; IsntNull(ssig); ssig=ssig->next )
		{
			if( !OBJISSIG(ssig) )
				continue;
			ssinfo = pgpSigToSigInfo( ssig );
			if( ssinfo->type != PGP_SIGTYPE_KEY_PROPERTY )
				continue;
			if( ssinfo->by != key )
				continue;
			p = pgpFetchObject( ssig, &len );
			if( IsNull(p) )
				continue;
			i = 0;
			do {
				PGPBoolean hashed;
				p = pgpSigFindSubpacket( p,
								SIGSUB_KEY_REVOCATION_KEY, i++, NULL, NULL,
								&hashed, NULL, &matches );
				if( !p  || !hashed )
					break;
				/*
				 * data is 1 byte of class, 1 of pkalg, 20 bytes of
				 * fingerprint.  Last 8 of 20 are keyid.
				 */
				if( pgpMemoryEqual( p+2+20-8, rskinfo->keyID, 8 ) &&
							p[1] == rskinfo->pkalg )
				{
					/* Test to see that the self-sig is valid */
					if( !pgpSigTried( ssig ) )
					{
						/* Validate the self-sig */
						PGPKeySetRef kset;
						PGPNewOneKeySet( ssig, &kset );
						if( IsntNull( kset ) )
						{
							pgpCheckKeySigs( kset, NULL, FALSE, FALSE, FALSE,
											 NULL, NULL, NULL );
							PGPFreeKeySet( kset );
						}
					}
					/* Only accept if self-sig is valid */
					if( pgpSigChecked( ssig ) )
						success = TRUE;
					/* No point in searching ssig further even if bad */
					break; /* out of while( i<matches) loop */
				}
			} while( i < matches  &&  !success );
			if( success )
				break;
		}
		/* Now verify revocation sig if we can */
		if( success )
		{
			/* Look for signing key in any keydb */
			PGPKeyDBObjRef signkey = NULL;
			PGPContextRef context = PGPPeekKeyDBObjContext( key );
			PGPKeyDBRef refdb = pgpContextGetFirstKeyDB(context);
			PGPKeyID keyid;

			pgpSigID8( rsig, NULL, &keyid );
			while( IsntNull( refdb )  &&  IsNull( signkey ) )
			{
				pgpGetKeyByKeyID( refdb, &keyid, FALSE, FALSE, &signkey );
				if( IsNull(signkey) )
					refdb = pgpKeyDBNextKeyDB( refdb );
			}
			if( IsntNull( signkey ) )
			{
				/* If found signkey, verify rev signature and remove if bad */
				pgpAssert( pgpKeyDBObjIsReal( signkey ) );
				if( !pgpSigTried( rsig ) )
				{
					/* Validate the revocation sig */
					PGPKeySetRef kset;
					PGPNewOneKeySet( rsig, &kset );
					if( IsntNull( kset ) )
					{
						if( refdb == PGPPeekKeySetKeyDB( kset ) )
							refdb = NULL;
						pgpCheckKeySigs( kset, refdb, FALSE, FALSE, FALSE,
										 NULL, NULL, NULL );
						PGPFreeKeySet( kset );
					}
				}
				if( pgpSigTried(rsig) && !pgpSigChecked(rsig) )
					success = FALSE;
			}
		}
		if( !success )
			pgpMarkKeyDBObjectDeleted( rsig );
	}
	return kPGPError_NoErr;
}



/*
 * Called after we have read in a PGP Key and all its children.  Check
 * it for various conditions.
 */
	PGPError
pgpKeyCleanup( PGPKeyDBObj *key, PGPBoolean trusted )
{
	return sCleanupThirdPartyRevocations( key, trusted );
}




	PGPError
pgpReadKeyFile (PGPKeyDB *kdb, PGPFile *f, PGPBoolean trusted)
{
	PGPEnv				*pgpEnv;
	PGPContextRef		context;
	PGPKeyDBObj *		topobj = NULL;
	PGPKeyDBObj *		lastobj = NULL;
	PGPKeyDBObj *		parobj = NULL;
	PGPFileOffset		pos;
	PGPSize				len;
	PGPUInt32			size;
	PGPUInt32			bodylen;
	PGPUInt32			maxbuflen = 0;
	PGPSize				len1;
	PGPInt32			pktbyte;
	PGPByte *			buf = NULL;
	PGPError			err = kPGPError_NoErr;

	context = PGPPeekKeyDBContext( kdb );
	pgpEnv = pgpContextGetEnvironment( context );

	/* Skip until we get to a key packet */
	pos = pgpFileTell( f );
	while ((pktbyte = pgpPktByteGet(f, &len, &len1, &pos)) > 0) {
		if( PKTBYTE_TYPE(pktbyte) == PKTBYTE_PUBKEY
			|| PKTBYTE_TYPE(pktbyte) == PKTBYTE_SECKEY ) {
			pgpFileSeek( f, pos, SEEK_SET );
			break;
		}
		pos = pgpFileTell( f );
	}
	/* If no key packets, return empty-handed */
	if( pktbyte < 0 )
		goto error;
									   

	/* Read each packet, add to kdb, remember where it came from */
	pos = pgpFileTell( f );
	while ((pktbyte = pgpPktByteGet(f, &len, &len1, &pos)) > 0) {
		if( len > PGPOBJ_MAXSIZE )
		{
			err = pgpFileSeek( f, len, SEEK_CUR );
			if( IsPGPError( err ) )
				goto error;
			continue;
		}
		bodylen = len;
		len += pgpPktHeaderLen( pktbyte, len );
		if( len > maxbuflen ) {
			void *vbuf = buf;
			err = pgpContextMemRealloc( context, &vbuf, len, 0 );
			buf = vbuf;
			if( IsPGPError( err ) )
			{
				err = pgpFileSeek( f, len, SEEK_CUR );
				if( IsPGPError( err ) )
					goto error;
				continue;
			}
			maxbuflen = len;
		}
		/* Put the pktbyte + length into the beginning of buf */
		pgpPktHeaderPut( buf, pktbyte, bodylen );
		size = pgpPktBodyRead( buf+len-bodylen, bodylen, len1, f );
		if( size != bodylen )
			/* Premature EOF, just return with what we have */
			break;
		size = len;

		switch (PKTBYTE_TYPE(pktbyte)) {

		  case PKTBYTE_PUBKEY:
		  case PKTBYTE_SECKEY:
			  /* Do some checks on previously read-in key certificate */
			  if( IsntNull( topobj ) )
				  pgpKeyCleanup( topobj, trusted );
			  topobj = pgpAddObject( buf, len, kdb, NULL, trusted, &err );
			  lastobj = parobj = topobj;
			  if( IsPGPError( err ) )
				  goto error;
			  break;
		  case PKTBYTE_PUBSUBKEY:
		  case PKTBYTE_SECSUBKEY:
		  case PKTBYTE_ATTRIBUTE:
		  case PKTBYTE_USERID:
			  if( IsNull(topobj) || !OBJISTOPKEY(topobj) )
				  break;
			  parobj = pgpAddObject( buf, len, kdb, topobj, trusted, &err);
			  lastobj = parobj;
			  if( IsPGPError( err ) )
				  goto error;
			  break;
		  case PKTBYTE_SIG:
			  if( IsNull(parobj) || IsNull(topobj) || !OBJISTOPKEY(topobj) )
				  break;
			  lastobj = pgpAddObject(buf, len, kdb, parobj, trusted, &err);
			  if( IsPGPError( err ) )
				  goto error;
			  break;
		  case PKTBYTE_CRL:
			  if(IsNull(parobj) || IsNull(topobj) || !OBJISTOPKEY(topobj) )
				  break;
			  if( !OBJISKEY( parobj ) )
				  parobj = topobj;
			  pgpAssert( OBJISKEY(parobj) );
			  lastobj = pgpAddObject(buf, len, kdb, parobj, trusted, &err);
			  if( IsPGPError( err ) )
				  goto error;
			  break;
		  case PKTBYTE_TRUST:
			  if( IsNull(lastobj) || !trusted )
				  break;
			  err = pgpTrustPacket( buf, len, kdb, lastobj, pgpEnv );
			  if( IsPGPError( err ) )
				  goto error;
			  break;
		  case PKTBYTE_COMMENT:
			  /* Skip this */
			  break;
		  default:
			  /* Unknown packet type; save it for output */
			  if( IsNull(parobj) )
				  break;
			  lastobj = pgpAddObject(buf, len, kdb, parobj, trusted, &err);
			  if( IsPGPError( err ) )
				  goto error;
			  break;
		}
		pos = pgpFileTell( f );
	}
 error:
	if( IsntNull( buf ) )
		PGPFreeData( buf );
	if( IsntNull( topobj ) )
		pgpKeyCleanup( topobj, trusted );
	
	return err;
}

/*
 * Add objects from back-end supplied key array.  If newobjs is false we do
 * lookups to merge them with existing keys; if true we know they are all
 * new objects and skip the lookup.
 */
	static PGPError
sAddFromKeyArray( PGPKeyDB *kdb, PGPKeyDBObj *parent, PGPKeyDBObj *prev,
	PGPUInt32 **keyArray, PGPUInt32 keyCount, PGPBoolean newobjs )
{
	PGPUInt32		i;
	PGPUInt32		id;
	PGPUInt32		flags;
	PGPUInt32		nchildren;
	PGPKeyDBObj *	obj;
	PGPError		err = kPGPError_NoErr;

	for( i = 0; i<keyCount; ++i ) {
		id = *(*keyArray)++;
		flags = *(*keyArray)++;
		if( !newobjs &&
			IsntPGPError( PGPFindNode( kdb->idToObj, id,
											 (PGPUserValue *)&obj ) ) )
		{
			/* Type should be same, except dummy top key may become subkey */
			pgpAssert( (flags & 0x3f) == (obj->objflags & 0x3f)
					   || ( (flags&0x3f)==kPGPKeyDBObjType_SubKey &&
							pgpKeyDBObjIsDummy(obj) ) );

			/*
			 * If prev is valid and we have a parent, make sure our order
			 * is correct
			 */
			if( prev != (PGPKeyDBObj *)-1UL && IsntNull( parent ) )
			{
				if( IsntNull( prev ) )
				{
					if( prev->next != obj )
					{
						if( OBJISTOPKEY( obj ) )
							pgpUnlinkKey( obj, kdb ); /* dummy to subkey */
						else
							pgpUnlinkChild( obj, parent );
						obj->next = prev->next;
						obj->up = parent;
						prev->next = obj;
					}
				} else {
					/* Null prev means we should be first child */
					if( parent->down != obj )
					{
						if( OBJISTOPKEY( obj ) )
							pgpUnlinkKey( obj, kdb ); /* dummy to subkey */
						else
							pgpUnlinkChild( obj, parent );
						obj->next = parent->down;
						obj->up = parent;
						parent->down = obj;
					}
				}
			}

			obj->objflags &= ~( kPGPKeyDBObjFlags_Deleted |
							    kPGPKeyDBObjFlags_Dummy |
							    kPGPKeyDBObjFlags_X509  | 0x3f );
			obj->objflags |= flags & ( kPGPKeyDBObjFlags_Deleted |
									   kPGPKeyDBObjFlags_Dummy |
									   kPGPKeyDBObjFlags_X509  | 0x3f );

			/* Keys may get info field changed, so force update */
			if( OBJISKEY(obj) )
				pgpKeyDBObjRefresh( obj, FALSE );

		} else {
			if( flags & kPGPKeyDBObjType_Key ) {
				obj = sNewKey( kdb );
				if ( obj == NULL )
				{
					err = kPGPError_OutOfMemory;
					goto error;
				}
				pgpLinkKey( obj, kdb );
			} else {
				if ( flags & kPGPKeyDBObjType_SubKey )
					obj = sNewSubKey( kdb );
				else if ( flags & kPGPKeyDBObjType_UserID )
					obj = sNewUserID( kdb );
				else if ( flags & kPGPKeyDBObjType_Signature )
					obj = sNewSig( kdb );
				else if ( flags & kPGPKeyDBObjType_CRL )
					obj = sNewCRL( kdb );
				else
					obj = sNewUnknown( kdb );
				if ( obj == NULL )
				{
					err = kPGPError_OutOfMemory;
					goto error;
				}
				pgpLinkChild( obj, parent );
			}
			obj->idinfo.id = id;
			obj->objflags = flags | kPGPKeyDBObjFlags_ID;
			PGPAddNode( kdb->idToObj, id, (PGPUserValue)obj );
		}
		if( !OBJISSIG( obj ) )
		{
			nchildren = *(*keyArray)++;
			err = sAddFromKeyArray( kdb, obj, NULL, keyArray,
									nchildren, newobjs );
			if( IsPGPError( err ) )
				goto error;
		}
		prev = obj;
	}
 error:
	return err;
}

	PGPError
pgpAddFromKeyArray( PGPKeyDB *kdb, PGPKeyDBObj *parent, PGPUInt32 *keyArray,
	PGPUInt32 keyCount, PGPBoolean newobjs )
{
	PGPError err;

	err = sAddFromKeyArray( kdb, parent, (PGPKeyDBObj *)-1UL,
							&keyArray, keyCount, newobjs );
	if( IsPGPError( err ) )
		return err;
	pgpKeyDBChanged( kdb, TRUE );
	return kPGPError_NoErr;
}



/*
 * Given a seckey, a keydb, and some other information, create a new
 * set of keydb objects.  Then self-sign the pubkey.  Return the
 * key object.
 */
	PGPKeyDBObj *
pgpCreateKeypair (PGPEnv const *env, PGPKeyDB *db, PGPSecKey *seckey,
		   PGPKeySpec *keyspec, char const *name, size_t namelen,
		   PGPKeySet const *adkset, PGPByte adkclass,
		   PGPKeySet const *rakset, PGPByte rakclass,
		   PGPByte const *passphrase, PGPSize pplen, PGPBoolean hashedPhrase,
		   PGPError *error)
{
	PGPKeyDBObj *keyobj=NULL, *nameobj;
	PGPKeyDBObj *tmpkey;
	PGPKeyInfo *kinfo;
	PGPSigSpec *sigspec = NULL;
	PGPKeyIter *rakIter = NULL;
	PGPKeyIter *adkIter = NULL;
	PGPByte const *prefAlg;
	PGPSize prefAlgLength;
	PGPByte const *prefKeyServ;
	PGPSize prefKeyServLength;
	PGPUInt32 keyFlags, keyServPrefs;
	PGPBoolean fKeyFlags, fKeyServPrefs;
	PGPUInt16 validity;
	PGPUInt32 nkeys;
	PGPError err;
	PGPUInt32 i;
	PGPBoolean keyv4;
	PGPBoolean secret = TRUE;
	PGPPubKey *pubkey = NULL;
	PGPByte pkalg;

	pgpAssert( error );
	*error = kPGPError_NoErr;

#define CHECKRETVAL(val, err) if (val) { *error = err; goto cleanup; }

	/* SECRET KEY stuff */

	keyv4 = pgpKeySpecVersion (keyspec) > PGPVERSION_3;
	pkalg = seckey->pkAlg;
	if( pgpSecKeyOnToken(seckey) )
	{
		secret = FALSE;
		pubkey = pgpSecKeyPubkey(seckey);
	}
	keyobj = pgpCreateKey (db, NULL, secret, pubkey, (secret?seckey:NULL),
						   keyspec, pkalg);
	CHECKRETVAL (!keyobj, kPGPError_OutOfMemory);
	pgpAssert(OBJISKEY(keyobj));
	kinfo = pgpKeyToKeyInfo( keyobj );
	pgpSecKeySetKeyID( seckey, kinfo->keyID );
	if( !secret )
	{
		pgpPubKeySetKeyID( pubkey, kinfo->keyID );
		pgpKeyOnToken( keyobj );
	}

	nameobj = pgpCreateUserID (db, keyobj, (PGPByte *) name, namelen);
	CHECKRETVAL (!nameobj, kPGPError_OutOfMemory);

	/* Self-Sign the new pubkey */
	if (pgpKeyAlgUse(pgpPkalgByNumber(pkalg)) & PGP_PKUSE_SIGN)
	{
		sigspec = pgpSigSpecCreate (env, keyobj, PGP_SIGTYPE_KEY_GENERIC);
		CHECKRETVAL (!sigspec, kPGPError_OutOfMemory);
		pgpSigSpecSetPassphrase( sigspec, passphrase, pplen, hashedPhrase );
		if (keyv4)
		{
			/* New keys self-sign with special info! */
			static PGPByte defaultPrefAlg[] = {kPGPCipherAlgorithm_CAST5,
											   kPGPCipherAlgorithm_IDEA,
											   kPGPCipherAlgorithm_3DES};
			PGPByte flagbuf[4];

			prefAlg = pgpKeySpecPrefAlgs (keyspec, &prefAlgLength);
			if( IsntNull( prefAlg ) )
			{
				pgpSigSpecSetPrefAlgs (sigspec, 0, prefAlg, prefAlgLength);
			} else {
				pgpSigSpecSetPrefAlgs (sigspec, 0, defaultPrefAlg,
									   sizeof(defaultPrefAlg));
			}
			prefKeyServ = pgpKeySpecPrefKeyserv (keyspec, &prefKeyServLength);
			if( IsntNull( prefKeyServ ) )
			{
				pgpSigSpecSetPrefKeyServ (sigspec, 0, prefKeyServ,
										  prefKeyServLength);
			}
			keyFlags = pgpKeySpecKeyflags (keyspec, &fKeyFlags);
			if( fKeyFlags )
			{
				for( i=0; i<sizeof(flagbuf); ++i )
					flagbuf[i] = (PGPByte)(keyFlags >> (i*8));
				pgpSigSpecSetKeyFlags( sigspec, 0, flagbuf, sizeof(flagbuf) );
			}
			keyServPrefs = pgpKeySpecKeyservPrefs (keyspec, &fKeyServPrefs);
			if( fKeyServPrefs )
			{
				for( i=0; i<sizeof(flagbuf); ++i )
					flagbuf[i] = (PGPByte)(keyServPrefs >> (i*8));
				pgpSigSpecSetKeyServPrefs( sigspec, 0, flagbuf,
										   sizeof(flagbuf) );
			}

			pgpSigSpecSetVersion (sigspec, PGPVERSION_4);
			pgpSigSpecSetPrimaryUserID (sigspec, 0, TRUE);
			if ((validity=pgpKeySpecValidity (keyspec)) != 0)
			{
				pgpSigSpecSetKeyExpiration (sigspec, 0,
					(PGPUInt32)validity*24*60*60);
				kinfo->validityperiod = validity;
			}
			/* Add any additional decryption packets requested */
			if (IsntNull (adkset) &&
					((void)PGPCountKeys((PGPKeySet *)adkset,&nkeys), nkeys>0))
			{
				if( IsPGPError( err = PGPNewKeyIterFromKeySet (
											(PGPKeySet *)adkset, &adkIter ) ) )
				{
					*error = err;
					goto cleanup;
				}
				while( IsntPGPError( PGPKeyIterNextKeyDBObj( adkIter,
							kPGPKeyDBObjType_Key, &tmpkey ) ) )
				{
					PGPByte adinfo[22];
					kinfo = pgpKeyToKeyInfo( tmpkey );
					adinfo[0] = adkclass;
					adinfo[1] = kinfo->pkalg;
					pgpKeyFingerprint20( tmpkey, adinfo+2 );
					*error = pgpSigSpecSetAdditionalRecipientRequest (sigspec,
									0, adinfo, sizeof(adinfo) );
					CHECKRETVAL (*error, *error);
				}
				PGPFreeKeyIter (adkIter);
				adkIter = NULL;
			}
		}
		*error = pgpSignObject( nameobj, sigspec );
		if (IsntPGPError(*error) && keyv4 && IsntNull (rakset) &&
					((void)PGPCountKeys((PGPKeySet *)rakset,&nkeys), nkeys>0))
		{
			/* Add any Revocation Authorization Keys requested */
			/* These go in a separate, irrevocable signature on the key */
			pgpSigSpecDestroy (sigspec);
			sigspec = pgpSigSpecCreate (env, keyobj, PGP_SIGTYPE_KEY_PROPERTY);
			CHECKRETVAL (!sigspec, kPGPError_OutOfMemory);
			pgpSigSpecSetPassphrase( sigspec, passphrase, pplen, hashedPhrase);
			if( IsPGPError( err = PGPNewKeyIterFromKeySet (
											(PGPKeySet *)rakset, &rakIter ) ) )
			{
				*error = err;
				goto cleanup;
			}
			while( IsntPGPError( PGPKeyIterNextKeyDBObj( rakIter,
						kPGPKeyDBObjType_Key, &tmpkey ) ) )
			{
				PGPByte krinfo[22];
				kinfo = pgpKeyToKeyInfo( tmpkey );
				/* Note that rakclass must have 0x80 set to be effective */
				krinfo[0] = rakclass;
				krinfo[1] = kinfo->pkalg;
				pgpKeyFingerprint20 ( tmpkey, krinfo+2 );
				*error = pgpSigSpecSetRevocationKey (sigspec, 0, krinfo,
													 sizeof(krinfo) );
				CHECKRETVAL (*error, *error);
			}
			PGPFreeKeyIter (rakIter);
			rakIter = NULL;
			/* Make this signature non-revocable */
			pgpSigSpecSetRevocable (sigspec, 0, FALSE);
			*error = pgpSignObject (keyobj, sigspec);
		}
	}

	/* This sets both key and name trust */
	pgpKeySetAxiomatic(keyobj);
	pgpKeyDBChanged( db, TRUE );

 cleanup:
	if (pubkey)
		pgpPubKeyDestroy (pubkey);
	if (sigspec)
		pgpSigSpecDestroy (sigspec);
	if (rakIter)
		PGPFreeKeyIter( rakIter );
	if (adkIter)
		PGPFreeKeyIter( adkIter );
	return keyobj;
}


/*
 * Given a (top level) key, a keyspec, and some other information,
 * create a subkey keydb object and put it under the keydb.
 * Then sign the pubkey using the master signature key.
 * seckey is the secret key coresponding to the master signature key,
 * subseckey is the secret key to be used for creating the new subkey.
 */
	PGPKeyDBObj *
pgpCreateSubkeypair (PGPKeyDBObj *keyobj, PGPEnv const *env,
	PGPSecKey *subseckey, PGPKeySpec *keyspec,
	PGPByte const *passphrase, PGPSize pplen, PGPBoolean hashedPhrase,
	PGPError *error)
{
	PGPKeyDB *db = PGPPeekKeyDBObjKeyDB( keyobj );
	PGPKeyDBObj *subkeyobj = NULL;
	PGPSigSpec *sigspec = NULL;
	PGPKeyInfo *kinfo;
	PGPUInt16 validity;
	PGPUInt32 keyFlags;
	PGPBoolean fKeyFlags;
	PGPUInt32 i;
	PGPBoolean secret = TRUE;
	PGPPubKey *subpubkey = NULL;
	PGPByte pkalg;

	pgpAssert( error );
	*error = kPGPError_NoErr;

	/*
	 * All subkeys are V4.
	 */
	pgpKeySpecSetVersion (keyspec, PGPVERSION_4);

	pkalg = subseckey->pkAlg;
	if( pgpSecKeyOnToken(subseckey) )
	{
		secret = FALSE;
		subpubkey = pgpSecKeyPubkey(subseckey);
	}

	/* Create the subkey object on the secret keyring */
	subkeyobj = pgpCreateKey (db, keyobj, secret, subpubkey,
							  (secret?subseckey:NULL), keyspec, pkalg);
	if (!subkeyobj)
	{
		*error = kPGPError_OutOfMemory;
		return NULL;
	}
	kinfo = pgpKeyToKeyInfo( subkeyobj );
	if( !secret )
	{
		pgpPubKeySetKeyID( subpubkey, kinfo->keyID );
		pgpKeyOnToken( subkeyobj );
	}
	pgpSecKeySetKeyID( subseckey, kinfo->keyID );

	/* Sign the encryption key with the master signature key */
	sigspec = pgpSigSpecCreate (env, keyobj, PGP_SIGTYPE_KEY_SUBKEY);
	if (!sigspec)
	{
		*error = kPGPError_OutOfMemory;
		return NULL;
	}
	pgpSigSpecSetPassphrase( sigspec, passphrase, pplen, hashedPhrase );

	/* New keys self-sign with special info */
	if ((validity=pgpKeySpecValidity (keyspec)) != 0)
	{
		pgpSigSpecSetKeyExpiration (sigspec, 0,
									(PGPUInt32)validity*24*3600);
		kinfo->validityperiod = validity;
	}
	keyFlags = pgpKeySpecKeyflags( keyspec, &fKeyFlags );
	if( fKeyFlags )
	{
		PGPByte flagbuf[4];
		for( i=0; i<sizeof(flagbuf); ++i )
			flagbuf[i] = (PGPByte)(keyFlags >> (i*8));
		pgpSigSpecSetKeyFlags( sigspec, 0, flagbuf, sizeof(flagbuf) );
	}

	*error = pgpSignObject ( subkeyobj, sigspec );
	if (*error)
		goto cleanup;

	pgpKeySetAxiomatic(subkeyobj);
	pgpKeyDBChanged( db, TRUE );

cleanup:

	if (subpubkey)
		pgpPubKeyDestroy (subpubkey);
	if (sigspec)
		pgpSigSpecDestroy (sigspec);

	return subkeyobj;		/* success */
}


/*
 * Signs an object (and its parents); then deposits the new signature
 * object in place on the set.
 */
	PGPError
pgpSignObject(PGPKeyDBObj *obj, PGPSigSpec *spec)
{
	PGPKeyDB *		kdb;
	PGPByte *		sig;
	PGPInt32		siglen;
	PGPKeyDBObj *	sigobj	= NULL;
	PGPContextRef	cdkContext;
	PGPSigInfo *	sinfo;

	kdb = PGPPeekKeyDBObjKeyDB( obj );
	cdkContext = PGPPeekKeyDBContext( kdb );

	/*
	 * Force signatures with newer-than-RSA keys to use version 4,
	 * and also if the hash is newer than those supported by PGP 5.0.
	 * A bug in PGP 5.0 won't let it load keyrings using version 2.6 sigs
	 * which have a hash not recognized by it.  
	 * The exception is revocation sigs on top-level keys.  A different
	 * bug in PGP 5.0 won't let it validate V4 sigs on top-level keys.
	 * The only sigs it tries to recognize on top-level keys are revocation
	 * sigs.  So those we leave as V3.
	 *
	 * Now we are also using V4 signatures when signing with an RSA key on
	 * a non-RSA key.  There is no backwards compatibility issue there
	 * so we'd rather move forward with the V4 signatures consistently.
	 */
	if( pgpSigSpecVersion (spec) == PGPVERSION_3 ) {
		PGPKeyDBObj *signkey = pgpSigSpecSeckey (spec);
		PGPByte hashtype = pgpSigSpecHashtype (spec);
		PGPByte sigtype = pgpSigSpecSigType (spec);
		PGPKeyInfo *kinfo;
		PGPKeyDBObj *topobj = obj;
		while (!OBJISTOPKEY (topobj))
			topobj = topobj->up;
		pgpAssert (OBJISTOPKEY(topobj));
		kinfo = pgpKeyToKeyInfo( topobj );
		if( ! (sigtype == PGP_SIGTYPE_KEY_REVOKE && OBJISTOPKEY(obj)) ) {
			PGPInt32 pkalg;
			pgpGetKeyNumber( signkey, kPGPKeyProperty_AlgorithmID, &pkalg );
			if( pkalg > kPGPPublicKeyAlgorithm_RSA
					|| kinfo->pkalg > kPGPPublicKeyAlgorithm_RSA
					|| hashtype > kPGPHashAlgorithm_RIPEMD160) {
				pgpSigSpecSetVersion( spec, PGPVERSION_4 );
			}
		}
	}

	siglen = pgpSignObj (obj, spec, &sig);
	if (siglen < 0)
		return (PGPError)siglen;
	if (siglen == 0)
	{
		pgpContextMemFree( cdkContext, sig);
		return kPGPError_OutOfMemory;
	}
	sigobj = pgpCreateSig (kdb, obj, sig, siglen);
	pgpContextMemFree( cdkContext, sig);
	if( IsNull( sigobj ) )
		return pgpKeyDBError( kdb );
	sinfo = pgpSigToSigInfo( sigobj );
	sinfo->trust = PGP_SIGTRUSTF_CHECKED_TRIED | kPGPKeyTrust_Complete;

	return sigobj ? kPGPError_NoErr : kPGPError_OutOfMemory;
}



/*
 * Create a new public key in a keydb.  secret is true for pub/sec key
 * pairs.  If parent is non-null, this is a subkey of that parent.
 * If secret is set, skey is used, else pkey is used.
 */
	PGPKeyDBObj *
pgpCreateKey(PGPKeyDB *db, PGPKeyDBObj *parent, PGPBoolean secret,
	PGPPubKey const *pkey, PGPSecKey const *skey, PGPKeySpec const *ks,
	PGPByte pkalg)
{
	PGPContextRef context;
	PGPKeyDBObj *obj;
	PGPByte *pktbuf;
	PGPSize len, prefixlen, headerlen;
	PGPError err;
	PGPByte pkttype;

	if( secret )
		pkttype = parent ? PKTBYTE_SECSUBKEY : PKTBYTE_SECKEY;
	else
		pkttype = parent ? PKTBYTE_PUBSUBKEY : PKTBYTE_PUBKEY;
	pgpAssert( secret ? IsNull(pkey) : IsNull(skey) );
	pgpAssert( secret ? IsntNull(skey) : IsntNull(pkey) );

	pkttype = PKTBYTE_BUILD( pkttype, 1 );

	/* Calculate packet size */
	prefixlen = pgpKeyBufferLength(ks, pkalg);
	len = prefixlen + (secret ? pgpSecKeyBufferLength(skey)
					   : pgpPubKeyBufferLength(pkey) );
	headerlen = pgpPktHeaderLen( pkttype, len );
	len += headerlen;
	if (len >= PGPOBJ_MAXSIZE) {
		pgpKeyDBSetError( db, kPGPError_TroubleKeyTooBig);
		return NULL;
	}

	/* Allocate space */
	context = PGPPeekKeyDBContext( db );
	pktbuf = pgpContextMemAlloc( context, len, 0 );
	if( IsNull(pktbuf) )
	{
		pgpKeyDBSetError( db, kPGPError_OutOfMemory );
		return NULL;
	}

	/* Fill buffer */
	pgpPktHeaderPut( pktbuf, pkttype, len-headerlen );
	pgpKeyToBuffer(pktbuf+headerlen, ks, pkalg);
	if( secret )
		pgpSecKeyToBuffer(skey, pktbuf+headerlen+prefixlen);
	else
		pgpPubKeyToBuffer(pkey, pktbuf+headerlen+prefixlen);

	/* Add to keyring (makes a copy of the data buffer) */
	obj = pgpAddObject( pktbuf, len, db, parent, TRUE, &err );
	PGPFreeData( pktbuf );

	return obj;
}

/*
 * Change the data associated with a secret key.
 */
	PGPError
pgpUpdateSecKeyData( PGPKeyDBObj *key, PGPSecKey const *skey )
{
	PGPContextRef context;
	PGPKeyDB *db;
	PGPByte *newpktbuf;
	PGPSize len, prefixlen, headerlen, newlen, newheaderlen;
	PGPByte const *kdata;
	PGPByte version;
	PGPError err = kPGPError_NoErr;

	pgpAssert( OBJISKEY( key ) );
	pgpAssert( pgpKeyIsSec( key ) );

	/* Token keys don't need updates */
	if( pgpKeyIsOnToken( key ) )
		return kPGPError_NoErr;

	db = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( db );
	kdata = pgpFetchObject( key, &len );

	/* Calculate packet size */
	headerlen = pgpPktBufferHeaderLen( kdata );
	version = kdata[headerlen];
	prefixlen = (version == PGPVERSION_4) ? 6 : 8;
	newlen = prefixlen + pgpSecKeyBufferLength(skey);
	newheaderlen = pgpPktHeaderLen( kdata[0], newlen );

	newlen += newheaderlen;
	if (newlen >= PGPOBJ_MAXSIZE)
		return kPGPError_TroubleKeyTooBig;

	/* Allocate space */
	newpktbuf = pgpContextMemAlloc( context, newlen, 0 );
	if( IsNull(newpktbuf) )
		return kPGPError_OutOfMemory;

	/* Fill buffer */
	pgpPktHeaderPut( newpktbuf, kdata[0], newlen-newheaderlen );
	pgpCopyMemory( kdata+headerlen, newpktbuf+newheaderlen, prefixlen);
	pgpSecKeyToBuffer(skey, newpktbuf+newheaderlen+prefixlen);

	/* Replace secret key data */
	err = pgpUpdateObjData( key, newpktbuf, newlen );
	PGPFreeData( newpktbuf );

	return err;
}


/*
 * Create a new object where we have the data in a buffer
 */
	static PGPKeyDBObj *
sCreateObj( PGPKeyDB *db, PGPKeyDBObj *parent, PGPByte pkttype,
	PGPByte const *data, PGPSize len )
{
	PGPContextRef context;
	PGPKeyDBObj *obj;
	PGPSize headerlen;
	PGPByte *pktbuf;
	PGPError err;

	/* Calculate packet size */
	headerlen = pgpPktHeaderLen( pkttype, len );
	len += headerlen;
	if (len >= PGPOBJ_MAXSIZE) {
		pgpKeyDBSetError( db, kPGPError_TroubleKeyTooBig);
		return NULL;
	}

	/* Allocate space */
	context = PGPPeekKeyDBContext( db );
	pktbuf = pgpContextMemAlloc( context, len, 0 );
	if( IsNull(pktbuf) )
	{
		pgpKeyDBSetError( db, kPGPError_OutOfMemory );
		return NULL;
	}

	/* Fill buffer */
	pgpPktHeaderPut( pktbuf, pkttype, len-headerlen );
	pgpCopyMemory( data, pktbuf+headerlen, len-headerlen );

	/* Add to keyring (makes a copy of the data buffer) */
	obj = pgpAddObject( pktbuf, len, db, parent, TRUE, &err );
	PGPFreeData( pktbuf );

	return obj;
}


/*
 * Create a new (non-attribute) userid in a keydb.  key is the parent key.
 */
	PGPKeyDBObj *
pgpCreateUserID( PGPKeyDB *db, PGPKeyDBObj *key, PGPByte const *data,
	PGPSize len )
{
	PGPByte pkttype;

	pkttype = PKTBYTE_BUILD( PKTBYTE_USERID, 0 );
	return sCreateObj( db, key, pkttype, data, len );
}


/*
 * Create a new attribute userid in a keydb.  key is the parent key.
 */
	PGPKeyDBObj *
pgpCreateAttribute( PGPKeyDB *db, PGPKeyDBObj *key, PGPByte attributeType,
	PGPByte const *data, PGPSize len )
{
	PGPByte pkttype;
	PGPContextRef context;
	PGPKeyDBObj *obj;
	PGPSize headerlen;
	PGPSize datalen;
	PGPByte *pktbuf;
	PGPError err;

	datalen = len;
	len += 6;			/* account for attribute type, length */

	pkttype = PKTBYTE_BUILD( PKTBYTE_ATTRIBUTE, 0 );

	/* Calculate packet size */
	headerlen = pgpPktHeaderLen( pkttype, len );
	len += headerlen;
	if (len >= PGPOBJ_MAXSIZE) {
		pgpKeyDBSetError( db, kPGPError_TroubleKeyTooBig);
		return NULL;
	}

	/* Allocate space */
	context = PGPPeekKeyDBContext( db );
	pktbuf = pgpContextMemAlloc( context, len, 0 );
	if( IsNull(pktbuf) )
	{
		pgpKeyDBSetError( db, kPGPError_OutOfMemory );
		return NULL;
	}

	/* Fill buffer */
	pgpPktHeaderPut( pktbuf, pkttype, len-headerlen );
	pktbuf[headerlen+0]		 = 0xff;
	pktbuf[headerlen+1]		 = (PGPByte) ((datalen+1) >> 24);
	pktbuf[headerlen+2]		= (PGPByte) ((datalen+1) >> 16);
	pktbuf[headerlen+3]		= (PGPByte) ((datalen+1) >>  8);
	pktbuf[headerlen+4]		= (PGPByte) ((datalen+1) >>  0);
	pktbuf[headerlen+5]		= attributeType;
	pgpCopyMemory( data, pktbuf+headerlen+6, datalen );

	/* Add to keyring (makes a copy of the data buffer) */
	obj = pgpAddObject( pktbuf, len, db, key, TRUE, &err );
	PGPFreeData( pktbuf );

	return obj;
}



/*
 * Create a new sig in a keydb.  parent is the parent object.
 */
	PGPKeyDBObj *
pgpCreateSig( PGPKeyDB *db, PGPKeyDBObj *parent,
	PGPByte const *sig, PGPSize len )
{
	PGPByte pkttype;

	/* PGP 2.6.2 requires sigs to have two byte headers */
	pkttype = PKTBYTE_BUILD( PKTBYTE_SIG, 1 );
	return sCreateObj( db, parent, pkttype, sig, len );
}


/*
 * Create a new CRL in a keydb.  parent is the parent object.
 */
	PGPKeyDBObj *
pgpCreateCRL( PGPKeyDB *db, PGPKeyDBObj *parent,
	PGPByte const *crl, PGPSize len )
{
	PGPByte pkttype;

	pkttype = PKTBYTE_BUILD( PKTBYTE_CRL, 0 );
	return sCreateObj( db, parent, pkttype, crl, len );
}







/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
